#ifndef ODDEVEN_SINGER_WEIGHTS_HPP
#define ODDEVEN_SINGER_WEIGHTS_HPP

/////////////////////////////////////////
// DECLARATIONS
////////////////////////////////////////

#include <array>
#include "Singer.hpp"
#include "WeightsBase.hpp"
#include "CheckResults.hpp"

/**
 * Keeps track of the weights of the lines in the plane for a given point set
 */
template<int Q>
class Weights : public WeightsBase<Singer<Q>::V> {

protected:
    static constexpr int V = Singer<Q>::V;

    using WeightsBase<V>::set;
    using WeightsBase<V>::weights;
    using WeightsBase<V>::hashedWeight;

public:

    void add(int pt);

    void remove(int pt);

    void toggle(int pt);

    // toggle all points on the given line
    void switchLine(int ln);

    // A point is redundant when all lines through that point have weight > 1
    bool isRedundant(int pt) const;

    // A point is superfluous iff all lines through it have weight ≥ 1
    bool isSuperfluous(int pt) const;

    // A set is 'minimal' if there is no point for which all lines have weight > 1
    bool isMinimal() const;

    // Compute and store the invariant values of all points of the set
    void computeInvariants();

    // Compute and store the invariant values of all points of the plane - not only the current set
    void computeAllInvariants();

    void printWeights() const;

    int size() const {
        return set.getSize();
    }

    // Does this point already lie on a line with a weight of Q/2?
    bool onHeavyLine(int pt) const;


protected:
    // Compute invariant for the point.
    int computeInvariant(int pt) const;

};


/////////////////////////////////////////
// DEFINITIONS
////////////////////////////////////////

// note: definition of line is different from the bsets project:
// incidence: pt +ln ∈ D

template<int Q>
void Weights<Q>::add(int pt) {
    set.add(pt);
    for (int el: Singer<Q>::D) {
        int ln = (V + el - pt) % V;
        weights[ln]++;
    }
}

template<int Q>
void Weights<Q>::remove(int pt) { // reverse what is done in add
    for (int el: Singer<Q>::D) {
        int ln = (V + el - pt) % V;
        weights[ln]--;
    }
    set.remove(pt);
}

template<int Q>
void Weights<Q>::toggle(int pt) {
    if (set.contains(pt)) {
        remove(pt);
    } else {
        add(pt);
    }
}

template<int Q>
void Weights<Q>::switchLine(int ln) {
    for (int el: Singer<Q>::D) {
        int pt = (V + el - ln) % V;
        toggle(pt);
    }
}



template<int Q>
bool Weights<Q>::isSuperfluous(int pt) const {
    for (int el: Singer<Q>::D) {
        int ln = (V + el - pt) % V;
        if (weights[ln] == 0) {
            return false;
        }
    }
    return true;
}

template<int Q>
bool Weights<Q>::isRedundant(int pt) const {
    for (int el: Singer<Q>::D) {
        int ln = (V + el - pt) % V;
        if (weights[ln] < 2) {
            return false;
        }
    }
    return true;
}

template<int Q>
bool Weights<Q>::isMinimal() const {
    // check whether any of the fat points is redundant
    for (int pt = 0; pt < V; pt++) {
        if (set.contains(pt) && isRedundant(pt)) {
            return false;
        }
    }
    return true;
}

template<int Q>
int Weights<Q>::computeInvariant(int pt) const {
    int total = 0;
    for (int el: Singer<Q>::D) {
        int ln = (V + el - pt) % V;
        total += hashedWeight(ln);
    }
    return total;
}

template<int Q>
void Weights<Q>::computeInvariants() {
    for (int pt = 0; pt < V; pt++) {
        if (set.contains(pt)) {
            this->invariants[pt] = computeInvariant(pt);
        }
    }
}

template<int Q>
void Weights<Q>::computeAllInvariants() {
    for (int pt = 0; pt < V; ++pt) {
        this->invariants[pt] = computeInvariant(pt);
    }
}

template<int Q>
void Weights<Q>::printWeights() const {
    std::cout << "WEIGHTS ";
    for (int w = 0; w <= Q + 1; ++w) {
        std::cout << w << ": ";
        for (int i = 0; i < V; ++i) {
            if (weights[i] == w) {
                std::cout << i << "-";
            }
        }
        std::cout << std::endl;
    }
}

template<int Q>
bool Weights<Q>::onHeavyLine(int pt) const {
    // for all lines through pt, check whether the weight is >= Q/2
    for (int el: Singer<Q>::D) {
        int ln = (V + el - pt) % V;
        if (weights[ln] >= Q / 2) {
            return true;
        }
    }
    return false;
}

#endif //ODDEVEN_SINGER_WEIGHTS_HPP
