#ifndef ODDEVEN_SINGER_SINGER_HPP
#define ODDEVEN_SINGER_SINGER_HPP

#include <array>
#include "FFel.hpp"
#include "../plane/Point.hpp"

template<int Q>
class Singer {

public:
    static const std::array<int, Q + 1> D;
    static constexpr int V = Q * Q + Q + 1;

    using FF = FFel<Q>;

    /**
     * Return the point with given index, as a point of the plane
     */
    static Point<Q> point(int index) {
        static std::array<Point<Q>, V> points = initPoints();
        return points[index];
    }

    /**
     * Return the index of the point with given sequence nr
     */
     static int index(int seqnr) {
         static std::array<int, V> indices = initIndices();
         return indices[seqnr];
     }

private:
    // image of (0,0,1) by Singer cycle is (a,b,c)
    static const FF a;
    static const FF b;
    static const FF c;

    static std::array<int, Q + 1> initD();

    static bool checkDiffset(std::array<int, Q + 1> &diffset);

    static std::array<Point<Q>, V> initPoints();
    static std::array<int, V> initIndices();

};

////////////////////////////////////////
// DEFINITIONS - GENERAL
////////////////////////////////////////

template<int Q>
std::array<int, Q + 1> Singer<Q>::initD() {
    std::array<int, Q + 1> result{};
    int pos{0};
    FF x = 1;
    FF y = 0;
    FF z = 0;
    for (int el = 0; pos <= Q; el++) {
        if (z == 0) {
            result[pos] = el;
            ++pos;
        }
        FF newx = y;
        FF newy = z;
        z = a * x + b * y + c * z;
        x = newx;
        y = newy;
    }
    if (checkDiffset(result)) {
        return result;
    } else {
        throw std::logic_error("Difference set does not satisfy required properties");
    }
}

template<int Q>
bool Singer<Q>::checkDiffset(std::array<int, Q + 1> &diffset) {
    std::array<bool, V> present{};
    for (const int i: diffset) {
        for (const int j: diffset) {
            present[(V + i - j) % V] = true;
        }
    }
    for (int i = 1; i < V; i++) {
        if (!present[i]) {
            return false;
        }
    }
    return true;
}

template<int Q>
std::array<Point<Q>, Singer<Q>::V> Singer<Q>::initPoints() {
    std::array<Point<Q>, V> result{};
    FF x(1);
    FF y(0);
    FF z(0);
    for (int i = 0; i < V; ++i) {
        if (x != 0) {
            result[i] = Point<Q>(FF(1), y / x, z / x);
        } else if (y != 0) {
            result[i] = Point<Q>(FF(0), FF(1), z / y);
        } else {
            result[i] = Point<Q>(FF(0), FF(0), FF(1));
        }
        FF newx = y;
        FF newy = z;
        z = a * x + b * y + c * z;
        x = newx;
        y = newy;
    }
    return result;
}

template <int Q>
std::array<int, Singer<Q>::V> Singer<Q>::initIndices() {
    std::array<int, V> result{};
    for (int i=0; i < V; ++i) {
        result[point(i).index()] = i;
    }
    return result;
}

////////////////////////////////////////
// DEFINITIONS - SPECIALIZATIONS
////////////////////////////////////////

// Generated by z = x + y
// 0, 2, 6  mod 7
template<> const FFel<2> Singer<2>::a(1);
template<> const FFel<2> Singer<2>::b(1);
template<> const FFel<2> Singer<2>::c(0);
template<> const std::array<int, 3>  Singer<2>::D = initD();

// Generated by z = x + y
// 0, 2, 8, 12  mod 13
template<> const FFel<3> Singer<3>::a(1);
template<> const FFel<3> Singer<3>::b(1);
template<> const FFel<3> Singer<3>::c(0);
template<> const std::array<int, 4>  Singer<3>::D = initD();

// Generated by z = a^2x + αy + z
// 0, 5, 7, 17, 20  mod 21
template<> const FFel<4> Singer<4>::a = FFel<4>::all()[3];
template<> const FFel<4> Singer<4>::b = FFel<4>::all()[2];
template<> const FFel<4> Singer<4>::c(1);
template<> const std::array<int, 5>  Singer<4>::D = initD();

// Generated by z = x + 3y
// 0, 2, 9, 13, 25, 30  mod 13
template<> const FFel<5> Singer<5>::a(1);
template<> const FFel<5> Singer<5>::b(3);
template<> const FFel<5> Singer<5>::c(0);
template<> const std::array<int, 6>  Singer<5>::D = initD();

// Generated by z = 3x + y + z
// 0, 12, 14, 20, 23, 30, 52, 56  mod 57
template<> const FFel<7> Singer<7>::a(3);
template<> const FFel<7> Singer<7>::b(1);
template<> const FFel<7> Singer<7>::c(1);
template<> const std::array<int, 8>  Singer<7>::D = initD();

// Generated by z = x + αy + z
// 0, 4, 11, 17, 20, 48, 50, 58, 72  mod 73
template<> const FFel<8> Singer<8>::a(1);
template<> const FFel<8> Singer<8>::b = FFel<8>::all()[2];
template<> const FFel<8> Singer<8>::c(1);
template<> const std::array<int, 9>  Singer<8>::D = initD();

// Generated by z = x + αy + z
// 0, 18, 21, 23, 31, 35, 64, 75, 84, 90  mod 91
template<> const FFel<9> Singer<9>::a(1);
template<> const FFel<9> Singer<9>::b = FFel<9>::all()[2];
template<> const FFel<9> Singer<9>::c(1);
template<> const std::array<int, 10>  Singer<9>::D = initD();

// Generated by z = x + 3y
// 0, 2, 11, 19, 33, 37, 80, 87, 93, 103, 108, 132 mod 133
template<> const FFel<11> Singer<11>::a(1);
template<> const FFel<11> Singer<11>::b(3);
template<> const FFel<11> Singer<11>::c(0);
template<> const std::array<int, 12>  Singer<11>::D = initD();

// Generated by z = 2x + y + 7z
// 0, 12, 34, 41, 58, 60, 78, 88, 111, 143, 168, 174, 179, 182 mod 183
template<> const FFel<13> Singer<13>::a(2);
template<> const FFel<13> Singer<13>::b(1);
template<> const FFel<13> Singer<13>::c(7);
template<> const std::array<int, 14>  Singer<13>::D = initD();

#endif //ODDEVEN_SINGER_SINGER_HPP
