#ifndef ODDEVEN_SINGER_LINESUMGENERATOR_HPP
#define ODDEVEN_SINGER_LINESUMGENERATOR_HPP

#include <array>
#include <iostream>
#include "PointSet.hpp"
#include "Weights.hpp"
#include "ColourablePlane.hpp"
#include "SingerGeometry.hpp"
#include "Archive.hpp"
#include "CheckResults.hpp"

template<int Q>
class LineSumGenerator {

    static constexpr int V = Q*Q+Q+1;

private:
    
    // lines currently being used
    PointSet<V> lines {};
    
    // weights for current point set
    Weights<Q> weights;

    ColourablePlane<V> colourablePlane = ColourablePlane<V>(SingerGeometry<Q>());
    
    // add and register a single line
    void add(int line) {
        lines.add(line);
        weights.switchLine(line);
    }
    
    // remove a single line
    void remove(int line) {
        weights.switchLine(line);
        lines.remove(line);
    }
    
    // recursively generate starting from the current set
    void gen();
    
    // output the currently generated set, return its size
    int shipSet() const;

    // is this a line that can be added to the generated set?
    bool isValid (int line) const {
        return weights[line] <= Q/2; // otherwise the point set will decrease in size
    }

    // can the currently generated set validly be extended?
    bool isValid() const {
        return true; // lines.getSize() < 10; // debugging
    }

    // is the given line canonical in the set (wrt. isomorph freeness)
    CheckResult checkMaybeCanonical(int line) {
        return weights.hasLargestWeight(line) ? UNKNOWN : NOT_CANONICAL; // TODO make this more precise, e.g., largestUniqueBigWeight
    }

    // nr of results generated
    long totalCount = 0;

    std::array<long, V + 1> counts = {0,}; // too large, but who cares?

public:

    // add a point to the initial point set
    void addPoint(int pt) {
        weights.add(pt);
    }

    void generate();

    void printStatistics() const;

};

template<int Q>
void LineSumGenerator<Q>::generate() {
    // start generation with a single line
//    add(0);
    gen();
//    remove(0);
}


template<int Q>
void LineSumGenerator<Q>::printStatistics() const {
    // duplicate from AbstractGenerator
    int start = 0;
    while (counts[start] == 0) {
        start++;
    }
    int stop = V;
    while (counts[stop] == 0) {
        stop--;
    }

    for (int i = start; i <= stop; i++) {
        std::cerr << "Size " << i << ": " << counts[i] << std::endl;
    }
    std::cerr << "Total: " << totalCount << std::endl;

}

template<int Q>
void LineSumGenerator<Q>::gen() {
    const PointSet<V> &pointSet = weights.getSet();
    totalCount++;
    counts[shipSet()] ++;
    Archive<ColourablePlane<V>, V> archive(colourablePlane);
    for (int line=0; line < V; ++line) {
        if (/*!lines.contains(line) && */ isValid(line)) { // indeed! the same line may need to be used twice
            add(line);
            if (isValid()) {
                switch (checkMaybeCanonical(line)) {
                    case CANONICAL: {
                        if (archive.add(pointSet, pointSet.getSize())) { // TODO better invariant?
                            gen();
                        }
                        break;
                    }
                    case UNKNOWN: {
                        colourablePlane.colour(pointSet);
                        SparseGraphPtr canonicalForm
                                = colourablePlane.isNautyCanonicalLine(line, weights.weightsData());
                        if (canonicalForm) { // yes, it is special!
                            if (archive.add(pointSet, pointSet.getSize(), // TODO better invariant?
                                            std::move(canonicalForm))) {
                                gen();
                            }
                        }
                        break;
                    }
                    case NOT_CANONICAL: {
                        // do not recurse
                        break;
                    }
                }

            }
            remove(line);
        }
    }
    
}


template<int Q>
int LineSumGenerator<Q>::shipSet() const {
/*  // for debugging
    lines.getSet().print('%');
    printf(" ");
*/
    const PointSet<V> &pointSet = weights.getSet();
    pointSet.print('%');
    printf("\n");
    return pointSet.getSize();
}

#endif //ODDEVEN_SINGER_LINESUMGENERATOR_HPP
