#ifndef ODDEVEN_SINGER_ABSTRACTGENERATOR_HPP
#define ODDEVEN_SINGER_ABSTRACTGENERATOR_HPP

#include <array>
#include "Weights.hpp"
#include "ColourablePlane.hpp"
#include "Archive.hpp"
#include "SingerGeometry.hpp"

/////////////////////////////////////////
// DECLARATIONS
////////////////////////////////////////

template<int Q>
class AbstractGenerator {

protected:

    // check whether the currently generated set should be shipped
    virtual bool shouldBeShipped() const  = 0;

    // should a further point be added?
    virtual bool shouldAddPoints() const = 0;

    // is this a valid subset of a possible result?
    // (This check is called immediately after adding a point)
    virtual bool isValid() const = 0;

    // is this a point that can be added to the current set so that it still remains valid?
    virtual bool isValid(int pt) const = 0;

protected:

    static constexpr int V = Q * Q + Q + 1;

    Weights<Q> weights{};

    ColourablePlane<V> colourablePlane = ColourablePlane<V>(SingerGeometry<Q>());

    /**
     * Print the current set to stdout and return the number of points in
     * the projective plane which it represents;
     */
    virtual int shipSet() const;

    // add a point (which is not already in the current set)
    virtual void add(int pt) {
        weights.add(pt);
    }

    // remove a point (which is already in the current set)
    virtual void remove(int pt) {
        weights.remove(pt);
    }

    // is the given point canonical in the set (wrt. isomorph freeness)
    CheckResult checkMaybeCanonical(int pt) {
        weights.computeInvariants();
        return weights.checkCanonical(pt);
    }

    void gen();

public:

    void generate();

    void processArgs(int argc, char *argv[]);

    // print statistics to stderr
    void printStatistics() const;

private:
    // nr of results generated
    long totalCount = 0;

    std::array<long, V + 1> counts = {0,}; // too large, but who cares?

protected:
    // Parameters to use when splitting process for parallelization
    int splitDepth = V + 1; // default = do not split
    int splitIndex = 0;
    int splitModulus = 0;
    int splitCounter = 0;
};

/////////////////////////////////////////
// DEFINITIONS
////////////////////////////////////////

template<int Q>
void AbstractGenerator<Q>::generate() {

    // start with a two point subset. All two point subsets are equivalent/isomorphic
    add(0);
    if (shouldBeShipped()) {
        totalCount++;
        counts[shipSet()]++;
    }
    add(1);
    gen();

    remove(1);
    remove(0); // these were added to be able to check memory leaks

}

template<int Q>
void AbstractGenerator<Q>::processArgs(int argc, char **argv) {
    if (argc > 1) {
        if (argc != 4) {
            std::cerr << "Usage: " << argv[0] << " depth index modulus";
        } else {
            splitDepth = (int) strtol(argv[1], nullptr, 10);
            splitIndex = (int) strtol(argv[2], nullptr, 10);
            splitModulus = (int) strtol(argv[3], nullptr, 10);
        }
    }
}

template<int Q>
int AbstractGenerator<Q>::shipSet() const {
    // print to stdout
    const PointSet<V> &set = this->weights.getSet();
    set.print('%');
    printf("\n");
//    weights.printWeights();
    return set.getSize();
}

template<int Q>
void AbstractGenerator<Q>::gen() {
    const PointSet<V> &currentSet = weights.getSet();
    // bail out when not in the correct search branch (when parallelized)
    if (currentSet.getSize() == splitDepth) {
        if (splitCounter++ % splitModulus != splitIndex) {
            return;
        }
    }

    if (shouldBeShipped()) {
        totalCount++;
        counts[shipSet()]++;
    }
    if (shouldAddPoints()) {
        Archive<ColourablePlane<V>, V> archive(colourablePlane);
        for (int pt = 0; pt < V; pt++) {
            if (!currentSet.contains(pt)) {
                if (isValid(pt)) {
                    add(pt);
                    if (isValid()) {
                        switch (checkMaybeCanonical(pt)) {
                            case CANONICAL: {
                                if (archive.add(currentSet, weights.computeSetInvariant())) {
                                    gen();
                                }
                                break;
                            }
                            case UNKNOWN: {
                                colourablePlane.colour(currentSet);
                                SparseGraphPtr canonicalForm
                                        = colourablePlane.isNautyCanonical(pt, weights.invariantsData());
                                if (canonicalForm) { // yes, it is special!
                                    if (archive.add(currentSet, weights.computeSetInvariant(),
                                                    std::move(canonicalForm))) {
                                        gen();
                                    }
                                }
                                break;
                            }
                            case NOT_CANONICAL: {
                                // do not recurse
                                break;
                            }
                        }
                    }
                    remove(pt);
                }
            }
        }
    }
}

template<int Q>
void AbstractGenerator<Q>::printStatistics() const {
    int start = 0;
    while (counts[start] == 0) {
        start++;
    }
    int stop = V;
    while (counts[stop] == 0) {
        stop--;
    }

    for (int i = start; i <= stop; i++) {
        std::cerr << "Size " << i << ": " << counts[i] << std::endl;
    }
    std::cerr << "Total: " << totalCount << std::endl;
}

#endif //ODDEVEN_SINGER_ABSTRACTGENERATOR_HPP
