#ifndef BSETS_PLANE_POINT_HPP
#define BSETS_PLANE_POINT_HPP

#include "FFel.hpp"

template<int Q>
class Projectivity;

////////////////////////////////////////
// DECLARATIONS
////////////////////////////////////////

// TODO may not be needed, only used in Singer

/**
 * Point of PG(2,Q). Also used for lines.
 */
template<int Q>
class Point {

    using FF = FFel<Q>;

public:
    FF x;
    FF y;
    FF z;

    /**
     * Point with given coordinates, not necessarily normalized
     */
    Point(const FF &x, const FF &y, const FF &z) : x(x), y(y), z(z) {}

    /**
     * 'point' with all zero coordinates, needed to initialize arrays. Avoid using this!
     */
    Point() : x(FF(0)), y(FF(0)), z(FF(0)) {}

    static const Point &get(int index) {
        return all()[index];
    }

    /**
     * Return the sequence number of the point. The point need not be normalized.
     * <ul>
     * <li> (0,0,1) → 0
     * <li> (0,1,*) → 1..Q
     * <li> (1,0,*) → Q+1..2Q, (1,*,*) → 2Q+1 → Q^2+Q
     * </ul>
     */
    int index() const;

    static constexpr int V = Q * Q + Q + 1;

    /**
     * All points of this plane (normalized), ordered by index
     */
    static const std::array<Point<Q>, V> &all() {
        static std::array<Point<Q>, V> result = initAll();
        return result;
    }

    /**
     * Apply a projectivity to this point
     */
    Point<Q> operator*(const Projectivity<Q> &projectivity) const;

    /**
     * Equality of points. Arguments need not be normalized. (Everything is equal to (0,0,0), but this should not be used.)
     */
    bool operator==(const Point<Q> &other) const;

    bool operator!=(const Point<Q> &other) const;

    Point conj() const {
        return Point(x.conj(), y.conj(), z.conj());
    }

    bool liesOn(const Point<Q> &line) const {
        return (x * line.x + y * line.y + z * line.z).isZero();
    }

    Point<Q> lineThrough(const Point<Q> &other) const;

private:

    static std::array<Point<Q>, V> initAll();

};

////////////////////////////////////////
// DEFINITIONS
////////////////////////////////////////

template<int Q>
bool Point<Q>::operator==(const Point<Q> &other) const {
    return x * other.y == y * other.x && x * other.z == z * other.x && y * other.z == z * other.y;
}

template<int Q>
bool Point<Q>::operator!=(const Point<Q> &other) const {
    return x * other.y != y * other.x || x * other.z != z * other.x || y * other.z != z * other.y;
}

template<int Q>
int Point<Q>::index() const {
    if (x.isZero()) {
        if (y.isZero()) {
            return 0;
        } else {
            return (z / y).index() + 1; // normalized
        }
    } else {
        if (x.isOne()) { // no need to normalize
            return (Q + 1) + y.index() * Q + z.index();
        } else {
            return (Q + 1) + (y / x).index() * Q + (z / x).index();
        }
    }
}

template<int Q>
std::array<Point<Q>, Point<Q>::V> Point<Q>::initAll() {
    std::array<Point<Q>, V> result{};

    int pos = 0;

    result[pos++] = Point{FF(0), FF(0), FF(1)};

    for (const auto &element: FF::all()) {
        result[pos++] = Point{FF(0), FF(1), element};
    }

    for (const auto &el1: FF::all()) {
        for (const auto &el2: FF::all()) {
            result[pos++] = Point(1, el1, el2);
        }
    }
    return result;
}

template<int Q>
Point<Q> Point<Q>::lineThrough(const Point<Q> &other) const {
    return Point<Q>(
            y * other.z - z * other.y,
            z * other.x - x * other.z,
            x * other.y - y * other.x
    );
}

#endif //BSETS_PLANE_POINT_HPP
