#ifndef ODDEVEN_LIB_WEIGHTSBASE_HPP
#define ODDEVEN_LIB_WEIGHTSBASE_HPP

#include <array>
#include "PointSet.hpp"
#include "CheckResults.hpp"

/////////////////////////////////////////
// DECLARATIONS
////////////////////////////////////////

/**
 * Common basis for the weights classes.
 */
template<int V>
class WeightsBase {

protected:
    std::array<int, V> weights{};
    PointSet<V> set{};
    std::array<int,V> invariants {};

public:

    // A set is a blocking set if all lines have weight ≥ 1
    bool isBlocking() const;

    const int *invariantsData() const {
        return invariants.data();
    }

    const int *weightsData() const {
        return weights.data();
    }

    // Determine canonicity using the computed invariants
    // The invariants must be computed before calling this method
    // (e.g., using computeInvariants in the derived class).
    CheckResult checkCanonical (int pt) const;

    // Compute the invariant for the complete set.
    // See restrictions on checkCanonical
    int computeSetInvariant() const;

    const PointSet<V> & getSet() const {
        return set;
    }

    // Return the size of the largest weight in the set
    int largest() const;

    // Does the given line has the largest of all weights?
    bool hasLargestWeight(int line) const;

    // Are all weights odd or even?
    bool isOddEven() const;

    int operator[] (int index) const {
        return weights[index];
    }

protected:

    int hashedWeight(int line) const {
        static std::array<int, V> hashes = initHashes(); // too many, but no access to Q here
        return hashes[weights[line]];
    }

private:
    static std::array<int, V> initHashes();

};

/////////////////////////////////////////
// DEFINITIONS
////////////////////////////////////////

template<int V>
bool WeightsBase<V>::isBlocking() const {
    for (const auto &weight: weights) {
        if (weight == 0) {
            return false;
        }
    }
    return true;
}

template<int V>
std::array<int, V> WeightsBase<V>::initHashes() {
    std::array<int, V> hashes{};
    int val = 1;
    for (int i = 1; i < V; i++) {
        hashes[i] = val;
        val *= 13;  // best odd
    }
    hashes[0] = val; // not used a lot
    return hashes;
}


template<int V>
bool WeightsBase<V>::hasLargestWeight(int line) const {
    int w = weights[line];
    for (int weight:weights) {
        if (w < weight) {
            return false;
        }
    }
    return true;
}


template<int V>
CheckResult WeightsBase<V>::checkCanonical(int pt) const {
    // if the point has the highest unique invariant, then it is canonical
    // if there are no unique invariants, the point has to have the highest
    // invariant value to be able to be canonical.

    // Step 1. Check for highest unique invariant
    //////

    std::array<bool, V> removed{};
    // this array keeps track of which points should still be visited,
    // after the loop, only for the unique invariants the value remains false

    int nrOfUnique = 0;
    for (int i = 0; i < V; i++) {
        if (set.contains(i) && !removed[i]) {
            bool found = false;
            for (int j = i + 1; j < V; j++) {
                if (set.contains(j) && !removed[j] && invariants[i] == invariants[j]) {
                    found = true;
                    removed[j] = true;
                }
            }
            if (found) {
                removed[i] = true;
            } else {
                // i has a unique invariant value
                nrOfUnique++;
                if (invariants[i] > invariants[pt]) {
                    return NOT_CANONICAL; // there is at least one unique invariant value, but it is larger than for the requested point
                }
            }
        }
    }
    if (nrOfUnique != 0) {
        if (removed[pt]) {
            return NOT_CANONICAL;
        } else {
            return CANONICAL;
        }
    }

    // Step 2. Highest invariant value
    /////
    for (int i = 0; i < V; i++) {
        if (set.contains(i) && invariants[i] > invariants[pt]) {
            return NOT_CANONICAL;
        }
    }

    return UNKNOWN; // must defer to Nauty...
}

template<int V>
int WeightsBase<V>::computeSetInvariant() const {
    int total = 0;
    for (int pt = 0; pt < V; pt++) {
        if (set.contains(pt)) {
            total += invariants[pt] * invariants[pt];
        }
    }
    return total;
}

template<int V>
int WeightsBase<V>::largest() const {
    int max = 0;
    for (int i = 0; i < V; i++) {
        if (weights[i] > max) {
            max = weights[i];
        }
    }
    return max;
}


template<int V>
bool WeightsBase<V>::isOddEven() const {
    int mod = weights[0] % 2;
    for (int weight: weights) {
        if (weight % 2 != mod) {
            return false;
        }
    }
    return true;
}

#endif //ODDEVEN_LIB_WEIGHTSBASE_HPP
