#ifndef ODDEVEN_LIB_SPARSEGRAPH_H
#define ODDEVEN_LIB_SPARSEGRAPH_H

#include <nausparse.h>
#include <cassert>
#include "PointLineGeometry.hpp"

/////////////////////////////////////////
// DECLARATIONS
////////////////////////////////////////

/**
 * Wraps a Nauty sparse graph into a C++ class. Allocates memory for the edge and degrees arrays.
 */
class SparseGraph : public sparsegraph {

public:

    /**
     * Create a sparse graph with the given nr of vertices and (directed) edges. Not initialized.
     */
    SparseGraph(int nrOfVertices, size_t nrOfDirectedEdges);

    /**
     * Create the sparse graph that corresponds to the given point line geometry
     */
    template <int V, int B>
    SparseGraph (const PointLineGeometry<V,B> &geometry);

    SparseGraph(const SparseGraph &) = delete;
    SparseGraph(SparseGraph &&);
    SparseGraph &operator = (const SparseGraph &) = delete;

    /**
     * Removes heap allocated memory
     */
    ~SparseGraph();

    // Do these represent the same graph?
    bool operator==( const SparseGraph &other) const;

    bool operator!=( const SparseGraph &other) const;
};

////////////////////////////////////////
// DEFINITIONS
////////////////////////////////////////

SparseGraph::SparseGraph(int nrOfVertices, size_t nrOfDirectedEdges) : sparsegraph() {
    //size_t vlen,dlen,elen,wlen;  // Sizes of arrays in units of type
    vlen = dlen = nv = nrOfVertices; // Number of vertices
    elen = nde = nrOfDirectedEdges; // Number of directed edges (loops contribute only 1)
    wlen = 0;

    d = new int[nrOfVertices]; // Array with out-degree of each vertex
    v = new size_t[nrOfVertices]; // Array of indexes into e[*]
    e = new int[nrOfDirectedEdges]; // Array to hold lists of neighbours
    w = nullptr; // Not implemented, should be NULL.
}

SparseGraph::SparseGraph(SparseGraph &&orig) {
    *((sparsegraph *)this) = *((sparsegraph *)&orig);
    orig.d = orig.e = nullptr;
    orig.v = nullptr;
    orig.w = nullptr;
}

SparseGraph::~SparseGraph() {
    delete[] d;
    delete[] v;
    delete[] e;
}

template <int V, int B>
SparseGraph::SparseGraph(const PointLineGeometry<V,B> &geometry) : SparseGraph(V+B, 2*geometry.getNrOfFlags()) {

    // points are stored first, lines with offset nrOfPoints
    int vpos = 0;
    size_t epos = 0;
    // point incident with line
    for (int pt = 0; pt < V; ++pt) {
        v[vpos] = epos;
        for (int ln = 0; ln < B; ++ln) {
            if (geometry.isIncident(pt, ln)) {
                e[epos] = ln + V;
                epos++;
            }
        }
        d[vpos] = (int) (epos - v[vpos]);
        vpos++;
    }
    // line incident with point
    for (int ln = 0; ln < B; ++ln) {
        v[vpos] = epos;
        for (int pt = 0; pt < V; ++pt) {
            if (geometry.isIncident(pt, ln)) {
                e[epos] = pt;
                epos++;
            }
        }
        d[vpos] = (int) (epos - v[vpos]);
        vpos++;
    }
    assert (nde == epos);
}

bool SparseGraph::operator==(const SparseGraph &other) const {
    return aresame_sg((sparsegraph *) this, (sparsegraph *) &other);
}

bool SparseGraph::operator!=(const SparseGraph &other) const {
    return !aresame_sg((sparsegraph *) this, (sparsegraph *) &other);
}



// UTILITY FUNCTIONS
// TODO clean up

#include "gtools.h"

/**
 * Converts a sparse graph to a string that can be printed.
 *
 * To be used with care: changes the internal representation of the graph and
 * returns a string that is fixed in memory and therefore should be used immediately (or copied)
 */
char * toString (sparsegraph *sg) {
    sortlists_sg (sg);
    char *result = sgtos6(sg); // contains a \n at the end !? - length stored in a variable sglen which is not accessible :-(
    result[strlen(result)-1] = '\0'; // remove \n
    return result;
}

#endif //ODDEVEN_LIB_SPARSEGRAPH_H