#ifndef ODDEVEN_LIB_POINTSET_H
#define ODDEVEN_LIB_POINTSET_H

#include <array>

/////////////////////////////////////////
// DECLARATIONS
////////////////////////////////////////

template<int V>
class PointSet {

private:
    // is point i present in this set?
    std::array<bool, V> present{};

    // size of the set currently being generated
    int size = 0;

public:

    // Add a point. Should only be called when the point is not already in the set.
    void add(int pt);

    // Remove a point that was previously added. Should only be called when the point is already in the set. This is not checked.
    void remove(int pt);

    // Remove the point if it is already there, otherwise add it
    void toggle(int pt);

    // Does this set contain the given point?
    bool contains(int pt) const { return present[pt]; }

    // How many points does this set contain?
    int getSize() const { return size; }

    // comparison (lexical order - 'smaller' sets have more points of low index)
    int cmp(const PointSet &other) const;

    // set equality
    bool operator==(const PointSet &other) const {
        return cmp(other) == 0;
    }

    bool operator!=(const PointSet &other) const{
        return cmp(other) != 0;
    }

    // lexical order
    bool operator<(const PointSet &other) const {
        return cmp(other) < 0;
    }

    // print to stdout - point numbers are prefixed with the given character
    void print(char prefix) const;

};

////////////////////////////////////////
// DEFINITIONS
////////////////////////////////////////

template <int V>
void PointSet<V>::add(int pt) {
    present[pt] = true;
    size++;
}

template <int V>
void PointSet<V>::remove(int pt) {
    // undo what was done in add
    size--;
    present[pt] = false;
}

template <int V>
void PointSet<V>::toggle(int pt) {
    if (present[pt]) {
        size --;
        present[pt] = false;
    } else {
        present[pt] = true;
        size ++;
    }
}

template <int V>
int PointSet<V>::cmp(const PointSet &other) const {
    // optimization by first looking at size is not useful because only
    // sets of the same size will be compared
    for (int pt = 0; pt < V; pt++) {
        if (present[pt]) {
            if (!other.present[pt]) {
                return 1;
            }
        } else {
            if (other.present[pt]) {
                return -1;
            }
        }
    }
    return 0;
}

template <int V>
void PointSet<V>::print(char prefix) const {
    for (int pt = 0; pt < V; ++pt) {
        if (present[pt]) {
            printf ("%c%d", prefix, pt);
        }
    }
}

#endif //ODDEVEN_LIB_POINTSET_H
