#ifndef ODDEVEN_LIB_FF_H
#define ODDEVEN_LIB_FF_H

#include <array>
#include <iostream>

////////////////////////////////////////
// DECLARATIONS
////////////////////////////////////////

template<int Q>
class FFel {

private:
    unsigned char value; // smaller than int

private:
    // constructor for internal use only, parameter v must be in the range 0..Q-1
    constexpr FFel(char v, bool) : value(v) {}

public:

    /**
     * Converts an integer to a finite field element representing the same value. Can be used
     * as implicit conversion
     */
    FFel(int v); // NOLINT(google-explicit-constructor)

    FFel() = default;

    FFel operator+(const FFel<Q> &other) const {
        static const std::array<std::array<FFel<Q>, 32>, Q> addition = initAddition();
        return addition[value][other.value];
    }

    FFel operator-(const FFel<Q> &other) const {
        static const std::array<std::array<FFel<Q>, 32>, Q> subtraction = initSubtraction();
        return subtraction[value][other.value];
    }

    FFel operator-() const {
        return zero - *this;
    }

    FFel operator*(const FFel<Q> &other) const {
        static const std::array<std::array<FFel<Q>, 32>, Q> multiplication = initMultiplication();
        return multiplication[value][other.value];
    }

    FFel inverse() const {
        return FFel(1, true) / *this;
    }

    FFel operator/(const FFel<Q> &other) const {
        static const std::array<std::array<FFel<Q>, 32>, Q> division = initDivision();
        return division[value][other.value];
    }

    bool operator==(const FFel<Q> &other) const {
        return value == other.value;
    }

    bool operator!=(const FFel<Q> &other) const {
        return value != other.value;
    }

    // should be safe to use  == 0 instead
    bool isZero() const {
        return value == 0;
    }

    // should be safe to use == 1 instead
    bool isOne() const {
        return value == 1;
    }

    /**
     * Convert to integer. Important! This only works for prime fields
     */
    explicit operator int() const {
        return value; // note that this only works for prime fields!
    }

    /**
     * Conjugate, i.e., image of the Frobenius map. Returns the identity when field is prime.
     */
    FFel conj() const {
        return *this;
    }

    /**
     * Return the internal index used to represent this element. We have all()[el.index()] == el
     */
    int index() const {
        return value;
    }

    static const FFel<Q> zero;
    static const FFel<Q> one;

    static const int CHARACTERISTIC;

    /**
     * All elements in this field
     */
    static std::array<FFel<Q>, Q> &all() {
        static std::array<FFel<Q>, Q> result = initAll();
        return result;
    };

    /**
     * Element with given index
     */
    static FFel get(int index) {
        return all()[index];
    }

private:

    // lookup tables for elementary operations, unfortunately not const as they need to be initialized later
    // using 32 for faster indexation (but then Q <= 32 must hold - cf initialize)
    static std::array<std::array<FFel<Q>, 32>, Q> initAddition();

    static std::array<std::array<FFel<Q>, 32>, Q> initSubtraction();

    static std::array<std::array<FFel<Q>, 32>, Q> initMultiplication();

    static std::array<std::array<FFel<Q>, 32>, Q> initDivision();

    static constexpr bool fields_supported[]{
            false, false, true, true, true, true, false, true,
            true, true, false, true, false, true, false, false,
            false, true, false, true, false, false, false, true,
            false, false, false, false, false, true, false, true,
            false
    };

    static std::array<std::array<FFel<Q>, 32>, Q> initAdditionNonPrime(const int (&add_table)[Q][Q]);

    static std::array<std::array<FFel<Q>, 32>, Q> initMultiplicationNonPrime();

    static std::array<FFel<Q>, Q> initAll();

};

////////////////////////////////////////
// DEFINITIONS - GENERAL
////////////////////////////////////////

template<int Q>
const FFel<Q> FFel<Q>::zero(0, true);

template<int Q>
const FFel<Q> FFel<Q>::one(1, true);

////////////////////////////////////////
// DEFINITIONS - PRIME ORDER
////////////////////////////////////////

template<int Q>
std::array<std::array<FFel<Q>, 32>, Q> FFel<Q>::initAddition() {
    std::array<std::array<FFel<Q>, 32>, Q> result{};
    if constexpr (fields_supported[Q]) {
        for (int i = 0; i < Q; i++) {
            for (int j = 0; j < Q; j++) {
                result[i][j] = FFel((i + j) % Q, true);
            }
        }
    } else {
        std::cerr << "Field of order " << Q << " not supported" << std::endl;
        exit(1);
    }
    return result;
}

template<int Q>
std::array<std::array<FFel<Q>, 32>, Q> FFel<Q>::initSubtraction() {
    // derive subtraction from addition already defined
    std::array<std::array<FFel<Q>, 32>, Q> result{};
    for (int i = 0; i < Q; i++) {
        for (int j = 0; j < Q; j++) {
            result[(FFel<Q>(i, true) + FFel<Q>(j, true)).index()][j] = FFel(i, true);
        }
    }
    return result;
}

template<int Q>
std::array<std::array<FFel<Q>, 32>, Q> FFel<Q>::initMultiplication() {
    std::array<std::array<FFel<Q>, 32>, Q> result{};
    if constexpr (fields_supported[Q]) {
        for (int i = 1; i < Q; i++) {
            for (int j = 1; j < Q; j++) {
                result[i][j] = FFel((i * j) % Q, true);
            }
        }
    } else {
        throw std::logic_error("Field of order " + std::to_string(Q) + " not supported");
    }
    return result;
}

template<int Q>
std::array<std::array<FFel<Q>, 32>, Q> FFel<Q>::initDivision() {
    // derive division from multiplication already defined
    std::array<std::array<FFel<Q>, 32>, Q> result{};
    for (int i = 1; i < Q; i++) {
        for (int j = 1; j < Q; j++) {
            result[(FFel<Q>(i, true) * FFel<Q>(j, true)).index()][j] = FFel(i, true);
        }
    }
    return result;
}

template<int Q>
const int FFel<Q>::CHARACTERISTIC = Q;

template<int Q>
std::array<FFel<Q>, Q> FFel<Q>::initAll() {
    std::array<FFel<Q>, Q> result;
    for (int i = 0; i < Q; i++) {
        result[i] = FFel<Q>(i, true);
    }
    return result;
}

template<int Q>
FFel<Q>::FFel(int v) {
    if (v >= 0) {
        value = v % Q;
    } else {
        value = Q + v % Q;
    }
}


////////////////////////////////////////
// DEFINITIONS - NONPRIME ORDER
////////////////////////////////////////

template<int Q>
std::array<std::array<FFel<Q>, 32>, Q> FFel<Q>::initMultiplicationNonPrime() {
    std::array<std::array<FFel<Q>, 32>, Q> result{};
    for (int i = 1; i < Q; i++) {
        for (int j = 1; j < Q; j++) {
            result[i][j] = FFel((i + j - 2) % (Q - 1) + 1, true);
        }
    }
    return result;
}


template<int Q>
std::array<std::array<FFel<Q>, 32>, Q> FFel<Q>::initAdditionNonPrime(const int (&add_table)[Q][Q]) {
    std::array<std::array<FFel<Q>, 32>, Q> result{};
    for (int i = 0; i < Q; i++) {
        for (int j = 0; j < Q; j++) {
            result[i][j] = FFel(add_table[i][j], true);
        }
    }
    return result;
}

template<>
constexpr int FFel<4>::CHARACTERISTIC = 2;

template<>
FFel<4>::FFel(int v) {
    value = (v % 2) != 0;
}

template<>
constexpr int FFel<8>::CHARACTERISTIC = 2;

template<>
FFel<8>::FFel(int v) {
    value = (v % 2) != 0;
}

template<>
constexpr int FFel<9>::CHARACTERISTIC = 3;

template<>
FFel<9>::FFel(int v) {
    static constexpr char table[]{1, 5, 0, 1, 5};
    value = table[v % 3 + 2];
}

/* We use a primitive element α such that α^2 = α + 1
 * The field elements are numbered
 *     0, 1, α, α^2
 * to make the multiplication table cyclic.
 * The same elements, listed in the same order, can also be written as
 *     0, 1, α, α+1
 * which is used to set up the addition table
 */
template<>
std::array<std::array<FFel<4>, 32>, 4> FFel<4>::initAddition() {
    return FFel<4>::initAdditionNonPrime(
            {
                    {0, 1, 2, 3},
                    {1, 0, 3, 2},
                    {2, 3, 0, 1},
                    {3, 2, 1, 0},
            });
}

template<>
std::array<std::array<FFel<4>, 32>, 4> FFel<4>::initMultiplication() {
    return initMultiplicationNonPrime();
}

template<>
FFel<4> FFel<4>::conj() const {
    static constexpr FFel table[]{FFel(0, true), FFel(1, true), FFel(3, true), FFel(2, true)};
    return table[value];
}

/* We use a primitive element α such that α^3 = α + 1
 * The field elements are numbered
 *     0, 1, α, α^2, α^3, α^4, α^5, α^6
 * to make the multiplication table cyclic.
 * The same elements, listed in the same order, can also be written as
 *     0, 1, α, α^2, α+1, α^2+α, α^2+α+1, α^2+1
 * which is used to set up the addition table
 */
template<>
std::array<std::array<FFel<8>, 32>, 8> FFel<8>::initAddition() {
    return FFel<8>::initAdditionNonPrime(
            {
                    {0, 1, 2, 3, 4, 5, 6, 7},
                    {1, 0, 4, 7, 2, 6, 5, 3},
                    {2, 4, 0, 5, 1, 3, 7, 6},
                    {3, 7, 5, 0, 6, 2, 4, 1},
                    {4, 2, 1, 6, 0, 7, 3, 5},
                    {5, 6, 3, 2, 7, 0, 1, 4},
                    {6, 5, 7, 4, 3, 1, 0, 2},
                    {7, 3, 6, 1, 5, 4, 2, 0}
            });
}

template<>
std::array<std::array<FFel<8>, 32>, 8> FFel<8>::initMultiplication() {
    return initMultiplicationNonPrime();
}

template<>
FFel<8> FFel<8>::conj() const {
    static constexpr FFel table[]{
            FFel(0, true), FFel(1, true), FFel(3, true), FFel(5, true),
            FFel(7, true), FFel(2, true), FFel(4, true), FFel(6, true),
    };
    return table[value];
}


/* We use a primitive element 1+α with α^2 = - 1
 * The field elements are numbered
 *     0, 1, 1+α, (1+α)^2, (1+α)^3, (1+α)^4, (1+α)^5, (1+α)^6, (1+α)^7
 * to make the multiplication table cyclic.
 * The same elements, listed in the same order, can also be written as
 *     0, 1, 1+α,      -α,     1-α,      -1,    -1-α,       α,    -1+α
 * which is used to set up the addition table
 */
template<>
std::array<std::array<FFel<9>, 32>, 9> FFel<9>::initAddition() {
    return FFel<9>::initAdditionNonPrime(
            {
                    {0, 1, 2, 3, 4, 5, 6, 7, 8},
                    {1, 5, 8, 4, 6, 0, 3, 2, 7},
                    {2, 8, 6, 1, 5, 7, 0, 4, 3},
                    {3, 4, 1, 7, 2, 6, 8, 0, 5},
                    {4, 6, 5, 2, 8, 3, 7, 1, 0},
                    {5, 0, 7, 6, 3, 1, 4, 8, 2},
                    {6, 3, 0, 8, 7, 4, 2, 5, 1},
                    {7, 2, 4, 0, 1, 8, 5, 3, 6},
                    {8, 7, 3, 5, 0, 2, 1, 6, 4}
            });
}

template<>
std::array<std::array<FFel<9>, 32>, 9> FFel<9>::initMultiplication() {
    return initMultiplicationNonPrime();
}

template<>
FFel<9> FFel<9>::conj() const {
    static constexpr FFel table[]{
            FFel(0, true), FFel(1, true), FFel(4, true),
            FFel(7, true), FFel(2, true), FFel(5, true),
            FFel(8, true), FFel(3, true), FFel(6, true),
    };
    return table[value];
}

////////////////////////////////////////
// DEFINITIONS - EVEN CHARACTERISTIC
////////////////////////////////////////

template<>
FFel<2> FFel<2>::operator-() const { return *this; }

template<>
FFel<4> FFel<4>::operator-() const { return *this; }

template<>
FFel<8> FFel<8>::operator-() const { return *this; }

template<>
FFel<16> FFel<16>::operator-() const { return *this; }


#endif //ODDEVEN_LIB_FF_H
