#ifndef ODDEVEN_LIB_COLOURABLEPLG_H
#define ODDEVEN_LIB_COLOURABLEPLG_H

#include <optional>
#include <memory>
#include "SparseGraph.hpp"
#include "PointLineGeometry.hpp"

/////////////////////////////////////////
// DECLARATIONS
////////////////////////////////////////

using SparseGraphPtr = std::unique_ptr<SparseGraph>;

/**
 * Allows a sparse graph based point line geometry to be coloured with the intention
 * of computing its canonical form.
 *
 * This class must be extended with a method to fill in the labels to be useful
 */
template<int V, int B>
class ColourablePLG : SparseGraph {

protected:
    int lab[V + B]{0,};
    int ptn[V + B]{0,};
    int orbits[V + B]{0,};

public:
    ColourablePLG(const PointLineGeometry<V, B> &geometry) : SparseGraph(geometry) {}

    // check whether the given point is canonical in the current set, according to Nauty
    // Returns the null pointer if not canonical. Returns the canonical form (for further use)
    // when the point is canonical.
    //
    // IMPORTANT
    // - Only call when the invariants for the points in the current set are computed, and stored in weights.
    // - pt must be known to have the highest invariant value of all points in the set
    SparseGraphPtr isNautyCanonical(int pt, const int invariants[]);

    // as isNautyCanonical but for a given line. Note that the invariants are still the point invariants
    SparseGraphPtr isNautyCanonicalLine (int line, const int invariants[]);

    // return the corresponding canonical form (after the points and lines have been coloured)
    SparseGraphPtr canonicalForm();

    // size of the stabilizer group (if small enough)
    int stabilizerSize();

    // print the orbits to stdout - only valid after a call to canonicalForm or stabilizerSize
    void printOrbits();
    
private:
    ColourablePLG() = default;

    ColourablePLG(const ColourablePLG &other) = delete;
    const ColourablePLG &operator = (const ColourablePLG &other) = delete;
};

////////////////////////////////////////
// DEFINITIONS
////////////////////////////////////////

template<int V, int B>
SparseGraphPtr ColourablePLG<V, B>::canonicalForm() {
    DEFAULTOPTIONS_SPARSEGRAPH(options);
    options.defaultptn = false; // use colours
    statsblk stats{0,};

    SparseGraphPtr canonsg{ new SparseGraph(nv, nde) };
    options.getcanon = true;
    sparsenauty(this, lab, ptn, orbits, &options, &stats, canonsg.get());
    return canonsg;
}

template<int V, int B>
int ColourablePLG<V,B>::stabilizerSize() {
    DEFAULTOPTIONS_SPARSEGRAPH(options);
    options.defaultptn = false; // use colours
    //options.writeautoms = true; // for debugging purposes
    statsblk stats{0,};
    sparsenauty(this, lab, ptn, orbits, &options, &stats, nullptr);
    return (int)stats.grpsize1;
}

template<int V, int B>
SparseGraphPtr ColourablePLG<V, B>::isNautyCanonical(int pt, const int invariants[]) {
    SparseGraphPtr canonsg = canonicalForm();

    // Check whether this point is mapped to the first orbit of the same invariant value
    // as pt. We know that this is the highest possible value, but this is not used.

    // because of the way things are coloured, the first size positions in lab
    // correspond to points of the set!
    int value = invariants[pt];
    int i = 0;
    while (i < V && invariants[lab[i]] != value) {
        i++;
    }
    if (orbits[pt] == orbits[lab[i]]) {
        return canonsg;
    } else {
        return SparseGraphPtr{};
    }
}

template<int V, int B>
SparseGraphPtr ColourablePLG<V, B>::isNautyCanonicalLine(int line, const int invariants[]) {
    SparseGraphPtr canonsg = canonicalForm();
    int value = invariants[line];
    int i = V;
    while (i < V+B && invariants[lab[i] - V] != value) {
        i++;
    }
    if (orbits[line + V] == orbits[lab[i]]) {
        return canonsg;
    } else {
        return SparseGraphPtr{};
    }
}


template<int V, int B>
void ColourablePLG<V,B>::printOrbits() {
    int nr = 0;
    int total = 0;
    // point orbits
    while (total < V) {
        // find next orbit
        while (orbits[nr] < nr) {
            nr++;
        }
        std::cout << "[ ";
        for (int pt = 0; pt < V; pt++) {
            if (orbits[pt] == nr) {
                std::cout << pt << " ";
                total++;
            }
        }
        std::cout << "] ";
        nr++;
    }
    std::cout << " :: ";
    // line orbits
    while (total < V + B) {
        // find next orbit
        while (orbits[nr] < nr) {
            nr++;
        }
        std::cout << "[ ";
        for (int ln = V; ln < V + B; ln++) {
            if (orbits[ln] == nr) {
                std::cout << (ln - V) << " ";
                total++;
            }
        }
        std::cout << "] ";
        nr++;
    }
}

#endif // ODDEVEN_LIB_COLOURABLEPLG_H
