#ifndef ODDEVEN_INVOL_ARCHIVE_H
#define ODDEVEN_INVOL_ARCHIVE_H

#include <vector>
#include <memory>
#include "PointSet.hpp"
#include "SparseGraph.hpp"

/////////////////////////////////////////
// DECLARATIONS
////////////////////////////////////////

using SparseGraphPtr = std::unique_ptr<SparseGraph>;

/**
 * Represents a set of non-equivalent point sets. Used during generation to
 * check whether a newly generated set is equivalent to a set that was generated earlier.
 *
 * Uses invariants to avoid unnecessary computation of the canonical form of its members.
 *
 * @tparam C class which computes the canonical form. Needs method .colour(set) and .canonicalForm
 * @tparam V parameter of the point sets PointSet<V> being stored in the archive
 */
template<class C, int V>
class Archive {

private:
    C &canonizer;

    // private class, represents a member of the archive - do not use separately
    class Member {
    public:
        const PointSet<V> set;

        int invariant;

        mutable SparseGraphPtr canonicalForm; // lazily computed

        Member(PointSet<V> set, int invariant, SparseGraphPtr canonicalForm)
                : set(set), invariant(invariant), canonicalForm(std::move(canonicalForm)) {};

        // Return the canonical form, lazily computed
        const SparseGraph &getCanonicalForm(C &cig) const;

        // Does this member represent a point set equivalent to the given point set?
        bool isSameAs(Member &other, C &cig) const;

    };

    std::vector<Member> members;

public:

    Archive(C &cig) : canonizer(cig) {

    }

    // Add the given point set to the archive, with the given invariant value and canonical form
    // If an equivalent point set is already in the archive, then no action is taken and false is returned, otherwise
    // the point set is added, and true is returned
    bool add(const PointSet<V> &set, int invariant, SparseGraphPtr canonicalForm = SparseGraphPtr());

    // size of the archive
    int size() const {
        return members.size();
    }
};

/////////////////////////////////////////
// DEFINITIONS
////////////////////////////////////////

template<class C, int V>
bool Archive<C, V>::Member::isSameAs(Member &other, C &cig) const {
    return invariant == other.invariant &&
           other.set.getSize() == set.getSize() &&
           getCanonicalForm(cig) == other.getCanonicalForm(cig);
}

template<class C, int V>
const SparseGraph &Archive<C, V>::Member::getCanonicalForm(C &cig) const {
    if (!canonicalForm) {
        cig.colour(set);
        canonicalForm = cig.canonicalForm();
    }
    return *canonicalForm;
}

template<class C, int V>
bool Archive<C, V>::add(const PointSet<V> &set, int invariant, SparseGraphPtr canonicalForm) {
    Member other(set, invariant, std::move(canonicalForm));
    for (Member &member: members) {
        if (other.isSameAs(member, canonizer)) {
            return false;
        }
    }
    members.emplace_back(std::move(other)); // emplace_back avoids one copy?
    return true;
}

#endif //ODDEVEN_INVOL_ARCHIVE_H
